#include "SimulationEngineController.h"

#include "../DataTypes/DataWheelCylinder.h"
#include "../DataTypes/DataCylinder.h"
#include "../DataTypes/Vector3f.h"
#include <time.h>
#include <sys/time.h>
#include <sys/timeb.h>
#include <sstream>

using namespace DataTypes;
namespace SimulationEngine
{

SimulationEngineController *SimulationEngineController::instance = 0;

SimulationEngineController::SimulationEngineController()
{
	/*Auto adjust parameters*/
	autoadjustmode = false;
	currentaveragecountsteps= 0;
	timelastadjust = 0;
	averagecountsteps = 100;
	lastaverageelapsedstep= 0.03;
	lastaveragestepsperiteration = 1;
	steplimit= 0.045;
	
	
	physicsfactory = PhysicsControllerFactory::GetInstance();
	timeb timeBuffer;
	ftime(&timeBuffer);
	lastSync = timeBuffer.time;
	lastSync += timeBuffer.millitm * 0.001;

	simulationTime=0;
	communication = new SimulationEngineCommunication();
	syncmode = false;
	stepmode = false;
	stepflag = false;
	initialized = false;
	fastmode = false;
	conf =ConfigurationServer::GetInstance();
	worldhardness = conf->GetDefaultPhysicsWorldHardness();
	worldbounciness = conf->GetDefaultPhysicsWorldBounciness();
	worldgravity.SetFirst(0);
	worldgravity.SetSecond(0);
	worldgravity.SetThird(conf->GetDefaultGravity());
	stepsize = conf->GetDefaultStepSize();
	syncwait = conf->GetSyncWait();
	physics = physicsfactory->GetPhysicsController();
	entitymanager = EntityManager::GetInstance();
	controlcommandmgr = ControlCommandManager::GetInstance();
	entitycommandmgr = EntityCommandManager::GetInstance();
	encoderdecoder = SimulationEngineEncoderDecoder::GetInstance();
	itercount=0;
	totaltime = 0;
	stepsperiteration = 1;
}

SimulationEngineController *SimulationEngineController::GetInstance()
{
	if (instance==0)
		instance =  new SimulationEngineController();
	return instance;
}

void SimulationEngineController::InitSimulation()
{
	initialized = true;
}

void SimulationEngineController::ResetSimulation()
{
	entitymanager->DestroyEntities();
	physics->ResetWorld();
}

bool SimulationEngineController::GetInitialized()
{
	return initialized;
}

void SimulationEngineController::SimulationLoop()
{
	/*Read new messages*/
	string message= communication->GetMessage();
	while (message.size()>0)
	{
		list <DataRequest *> reqs = encoderdecoder->DecodeMessage(message);
		std::list<DataRequest *>::iterator iter;
			
		iter = reqs.begin();
		while (iter!=reqs.end())
		{
			DataRequest *req= *iter;
			list<DataParameter *> response;
			if (DataControlRequest *datacr = dynamic_cast<DataControlRequest *>(req))
			{
				 response = controlcommandmgr->ExecuteControlCommand(datacr->GetCommandKey(), datacr->GetParameters());
			}	
			else if (DataEntityRequest *dataer = dynamic_cast<DataEntityRequest *>(req))
			{
				 response = entitycommandmgr->ExecuteEntityCommand(dataer->GetEntityId(), dataer->GetCommandKey(), dataer->GetParameters());
			}
			
			if (req->GetResponse())
			{
				string responsestring = encoderdecoder->EncodeMessage(response);
				communication->SendMessage("<Message>"+responsestring+"</Message>");					
			}			
			delete req;
			std::list<DataParameter *>::iterator iterresp;
			iterresp = response.begin();
			while (iterresp!=response.end())
			{
				DataParameter *todestroy = (*iterresp);
				delete todestroy;
				iterresp++;
			}
			
			iter++;
		}
		message= communication->GetMessage();
	}
	
	double elapsed = 0;
	
	/*End reading messages*/
	if (initialized)
	{
		bool stepworld = false;
		if (stepmode)
		{
			currentaveragecountsteps= 0;
			timelastadjust = 0;
		}
		if (syncmode)
		{
			timeb timeBuffer;
			ftime(&timeBuffer);
			double now = timeBuffer.time;
		    now += timeBuffer.millitm * 0.001;
			elapsed = now - lastSync;
			if (flagsync || elapsed>syncwait)
			{
				lastTime = now;
				if (!stepmode|| (stepmode && stepflag))
				{				
						if (elapsed<1000)
							totaltime+=elapsed;
						stepflag = false;
					 	stepworld = true;													
						lastSync = now;
						if (elapsed>syncwait)
						{
							stringstream itertext;
							itertext<< itercount;
							string tosend = "<Message><From_Simulation Command=\"SyncTimeout\"><Iteration>"+itertext.str()+ "</Iteration></FromSimulation></Message>";
							communication->SendMessage(tosend);
						}
				}
			}
		}
		else
		{
			if (!stepmode || (stepmode && stepflag))
			{
				timeb timeBuffer;
				ftime(&timeBuffer);
				double now = timeBuffer.time;
			    now += timeBuffer.millitm * 0.001;
			    
				elapsed = now - lastSync;
												
				totaltime+=elapsed;					
				lastSync = now;
				stepworld = true;					
				stepflag = false;
				/*timeb timeBuffer;
				ftime(&timeBuffer);
				double currentTime = timeBuffer.time;
			    currentTime += timeBuffer.millitm * 0.001;
				double elapsedSeconds = currentTime - lastTime;
				lastTime = currentTime;
				cout<<"Step size is:" << stepsize <<", and elapsed is"<<elapsedSeconds<<endl;*/
			}
		}
		if (stepworld)
		{
			if (!autoadjustmode || stepmode)
			{
				physics->StepWorld(stepsize, stepsperiteration, fastmode);
				simulationTime+= (stepsize*stepsperiteration);
			}
			else
			{				
				if (currentaveragecountsteps==0)
				{
					timeb timeBuffer;
					ftime(&timeBuffer);
					timelastadjust = timeBuffer.time;
					timelastadjust += timeBuffer.millitm * 0.001;
				}
				
				stringstream straux,straux1,straux2,straux3,straux4,straux5,straux6,straux7,straux8, straux9;

				currentaveragecountsteps++;

				
				//straux8<<averagecountsteps;
				//straux2<<currentaveragecountsteps; 
				//cout<<"currentaveragecountsteps:"<<straux2.str()<<" de "<<straux8.str()<<endl;
				
				if (currentaveragecountsteps==averagecountsteps)
				{
					timeb timeBuffer;
					ftime(&timeBuffer);
					double now = timeBuffer.time;
					now += timeBuffer.millitm * 0.001;
					//straux9<<timelastadjust;
					//cout<<"Time last adjust es "<<straux9.str();
					double currentstepselapsed = now- timelastadjust;
					//straux1<<currentstepselapsed;
					//cout<<"currentstepselapsed:"<<straux1.str()<<endl;
					//cout<<"Ajustando"<<endl;
					//promedio de timestep
					double average = currentstepselapsed/currentaveragecountsteps; 
					//straux3<<average;				
					//cout<<"average:"<<straux3.str()<<endl;
					currentaveragecountsteps = 0;
					//Voy a ajustar, steps per iteration
					lastaveragestepsperiteration = 1;
					double auxaverage=average;
					while (auxaverage>steplimit)
					{
						//cout<<"StepLimit superado"<<endl;
						//si el timestep esta muy grande, aumento el steps periteration
						lastaveragestepsperiteration += 1;
						auxaverage = average/lastaveragestepsperiteration;
					}
					lastaverageelapsedstep = auxaverage;
					straux4<<lastaverageelapsedstep;
					cout<<"Ajustado a lastaverageelapsedstep"<<straux4.str()<<endl;
					straux5<<lastaveragestepsperiteration;
					cout<<"Ajustado a lastaveragestepsperiteration"<<straux5.str()<<endl;
				}
				//straux6<<lastaverageelapsedstep;
				//cout<<"Avanzando lastaverageelapsedstep"<<straux6.str()<<endl;
				//straux7<<lastaveragestepsperiteration;
				//cout<<"Avanzando a lastaveragestepsperiteration"<<straux7.str()<<endl;
				physics->StepWorld(lastaverageelapsedstep, lastaveragestepsperiteration, fastmode);
				simulationTime+= (lastaverageelapsedstep*lastaveragestepsperiteration);
			}
			
			itercount++;
			string commandname;
			if (syncmode)
				commandname = "WorldBroadcastSync";
			else
				commandname = "WorldBroadcast";
			stringstream iterstr, tottime;
			iterstr<< itercount;
			tottime<<simulationTime;
			string worlddata = "<"+commandname+"><Iteration>"+iterstr.str()+ "</Iteration><Simulation_Time>"+tottime.str() +"</Simulation_Time>" +entitymanager->GetEntitiesString()+physics->GetDrawData() + "</"+commandname+">";
			communication->SendMessage(worlddata);
			flagsync = false;
		}
	}
}

bool SimulationEngineController::GetSyncMode()
{
	return syncmode;
}

bool SimulationEngineController::GetAutoAdjustMode()
{
	return autoadjustmode;	
}

void SimulationEngineController::RunEngine(bool gui)
{
	cout<< "Initialization..."<<endl;
	physics->InitWorld();
	cout<< "Running Simulation..."<<endl;
	
	timeb timeBuffer;
	ftime(&timeBuffer);
	lastTime = timeBuffer.time;
	lastTime += timeBuffer.millitm * 0.001;

	if (!gui)
	{
		while(1)
		{
			SimulationLoop();
		}
	}
	else
	{
		dsSimulationLoop (640,480);
	}
}

void SimulationEngineController::RfromQ(float *R, DataQuaternion quat)
{
	float q[4]={quat.GetFirst(),quat.GetSecond(),quat.GetThird(),quat.GetFourth()};
	float qq1 = 2*q[1]*q[1];
	float qq2 = 2*q[2]*q[2];
	float qq3 = 2*q[3]*q[3];
	R[0] = 1 - qq2 - qq3; 
	R[1] = 2*(q[1]*q[2] - q[0]*q[3]);
	R[2] = 2*(q[1]*q[3] + q[0]*q[2]);
	R[3] = 0;
	R[4] = 2*(q[1]*q[2] + q[0]*q[3]);
	R[5] = 1 - qq1 - qq3;
	R[6] = 2*(q[2]*q[3] - q[0]*q[1]);
	R[7] = 0;
	R[8] = 2*(q[1]*q[3] - q[0]*q[2]);
	R[9] = 2*(q[2]*q[3] + q[0]*q[1]);
	R[10] = 1 - qq1 - qq2;
	R[11] = 0;
}

void SimulationEngineController::simLoop (int pause)
{
	SimulationLoop();
	map<int, IPhysicalBody *> bodies = physics->GetBodies();
	std::map<int, IPhysicalBody*>::iterator iter;
			
	iter = bodies.begin();
	while (iter!=bodies.end())
	{
		IPhysicalBody *current= iter->second;		
		float rot[12];
		this->RfromQ(rot, current->GetQuaternion());
		Vector3f position = current->GetPosition();
		float pos[3]= {position.GetFirst(), position.GetSecond(),position.GetThird()};
		if (IPhysicalBox *currentbox = dynamic_cast<IPhysicalBox *>(current))
		{
			Vector3f size = currentbox->GetSize();
			float boxsize[3]= {size.GetFirst(),size.GetSecond(),size.GetThird()};
			dsDrawBox (pos,rot,boxsize);
		}
		else if (IPhysicalSphere *currentsphere = dynamic_cast<IPhysicalSphere *>(current))
		{
			float radius = currentsphere->GetRadius();
			dsDrawSphere(pos,rot,radius);
		}
		else if (IPhysicalWheelCylinder *currentcylinder = dynamic_cast<IPhysicalWheelCylinder *>(current))
		{
			float radius = currentcylinder->GetRadius();
			float length = currentcylinder->GetLength();
			dsDrawCylinder (pos,rot,length,radius);
		} 
		else if (IPhysicalCylinder *currentcylinder = dynamic_cast<IPhysicalCylinder *>(current))
		{
			float radius = currentcylinder->GetRadius();
			float length = currentcylinder->GetLength();
			dsDrawCylinder (pos,rot,length,radius);
		} 
		iter++;
	}
}

void SimulationEngineController::command (int cmd)
{

}


void SimulationEngineController::startSimulation()
{
  static float xyz[3] = {0.8317f,-0.9817f,0.8000f};
  static float hpr[3] = {121.0000f,-27.5000f,0.0000f};
  dsSetViewpoint (xyz,hpr);
}

void SimulationEngineController::FreeResources()
{
	communication->FreeResources();
	physics->FreeResources();
	entitymanager->DestroyEntities();
	controlcommandmgr->FreeResources();
	delete physics;
	delete communication;
	delete entitymanager;
	delete controlcommandmgr;
	delete entitycommandmgr;
	delete encoderdecoder;
	delete physicsfactory;
	
}

void SimulationEngineController::SetAutoAdjustMode(bool autoadj)
{
	this->autoadjustmode = autoadj;
	currentaveragecountsteps= 0;
	averagecountsteps = 100;
	lastaverageelapsedstep= 0.03;
	lastaveragestepsperiteration = 1;
	steplimit= 0.045;	
}

void SimulationEngineController::SetTimeStep(float timestep)
{
	this->stepsize = timestep;
}

void SimulationEngineController::SetStepMode(bool stepmd)
{
	this->stepmode = stepmd;
}

void SimulationEngineController::SetSyncMode(bool syncmode)
{
	this->syncmode = syncmode;
}

void SimulationEngineController::SetStepFlag(bool stepfg)
{
	this->stepflag = stepfg;
}

void SimulationEngineController::Synchronize(int iteration)
{
	if (itercount==iteration)
	{
		flagsync = true;
	}
}

void SimulationEngineController::SetSyncWait(double syncwait)
{
	this->syncwait = syncwait;
}

void SimulationEngineController::SetWorldHardness(float hardness)
{
	physics->SetWorldHardness(hardness);
	worldhardness = hardness;
}

void SimulationEngineController::SetWorldBounciness(float bounciness)
{
	physics->SetWorldBounciness(bounciness);
	worldbounciness = bounciness;
}

void SimulationEngineController::SetWorldGravity(float gravx, float gravy, float gravz)
{
	physics->SetWorldGravity(gravx,gravy,gravz);
	worldgravity.SetFirst(gravx);
	worldgravity.SetSecond(gravy);
	worldgravity.SetThird(gravz);
}

void SimulationEngineController::SetStepsPerIteration(int value)
{
	stepsperiteration = value;
}

void SimulationEngineController::SetFastMode(bool value)
{
	fastmode = value;
}

DataSimState SimulationEngineController::GetDataSimState()
{
	DataSimState toreturn(initialized, syncmode,flagsync,stepmode,stepflag,	broadcastflag, stepsize,lastTime,lastSync,	syncwait,itercount,worldgravity,worldhardness,worldbounciness, stepsperiteration, fastmode, autoadjustmode, lastaverageelapsedstep, lastaveragestepsperiteration);
	return toreturn;
}

void SimulationEngineController::SetState(DataSimState data)
{
	this->initialized = data.GetInitialized();
	this->syncmode = data.GetSyncMode();
	this->flagsync = data.GetFlagSync();
	this->stepmode = data.GetStepMode();
	this->stepflag = data.GetStepflag();
	this->broadcastflag = data.GetBroadcastflag();
	this->stepsize = data.GetStepSize();
	this->lastTime = data.GetLastTime();
	this->lastSync = data.GetLastSync();
	this->syncwait = data.GetSyncWait();
	this->itercount = data.GetIterCount();
	this->worldgravity = data.GetWorldGravity();
	this->worldhardness = data.GetWorldHardness();
	this->worldbounciness = data.GetWorldBounciness();
	this->stepsperiteration = data.GetStepsPerIteration();
	this->fastmode = data.GetFastMode();
	
	/*AutoAdjust*/
	this->autoadjustmode = data.GetAutoAdjustMode();
	this->lastaverageelapsedstep= data.GetLastAverageElapsedStep();
	this->lastaveragestepsperiteration = data.GetLastAverageStepsPerIteration();
	currentaveragecountsteps= 0;
	timelastadjust = 0;
	/*End AutoAdjust*/
}

string SimulationEngineController::GetPersistString()
{
	string toreturn = "<Simulation_Data>";
	toreturn += this->GetDataSimState().ToString();
	toreturn += this->entitymanager->GetEntitiesPersist();
	toreturn += "</Simulation_Data>";
	return toreturn;
}

}
